# GENERATE_ELLIPSE_LIB houses randomly generated as ellipse

# -------------------------------------------------------------------------------
# Copyright (c) 2016, Drexel University
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# * Redistributions of source code must retain the above copyright notice, this
#   list of conditions and the following disclaimer.
#
# * Redistributions in binary form must reproduce the above copyright notice,
#   this list of conditions and the following disclaimer in the documentation
#   and/or other materials provided with the distribution.
#
# * Neither the name of PixelRep nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# -------------------------------------------------------------------------------

# GetRandomEllipseImage() accepts [a] k-number of clusters (int). This function generates
# k-number of ellipse(s) with randomly initialized radii, rotation, and center of mass.
# The ellipse(s) is/are converted into a binary image. A binary image and the true boundary
# pixels
def GetRandomEllipseImage(K):
    import numpy as np

    # Initialize parameters of random ellipse
    IMSIZE = 1000
    mu1 = [500, 500]
    bw = np.zeros([IMSIZE, IMSIZE])
    ptsCombined = np.empty([1,2])
    pts = []
    ptsEdge = []
    bValidEllipse = False

    # Valid Ellipse Creation
    while bValidEllipse == False: # validity is dependent on overlap score of ellipse

        # Collect points and ellipse(s) edges of generated ellipse(s) centered at mu1
        for k in np.arange(0, K, 1, dtype=int):
            pts_tmp, ptsEdge_tmp = GetRandomEllipsePts(mu1, IMSIZE)
            ptsCombined = np.append(ptsCombined, pts_tmp, axis=0)
            pts.append(pts_tmp)
            ptsEdge.append(ptsEdge_tmp)

        # Offset ellipse(s) by randomly selecting a point of another ellipse as the new center of mass
        for k in np.arange(1, K, 1, dtype=int):
            ptsConcat = np.concatenate((pts[k], pts[k-1]))
            idxK = np.round((pts[k].shape[0] - 1) * np.random.uniform())
            idxCombined = np.round((ptsConcat.shape[0] - 1) * np.random.uniform())
            offset = ptsConcat[int(idxCombined), 0:] - pts[k][int(idxK), 0:]
            pts[k] = pts[k] + np.matlib.repmat(offset, pts[k].shape[0], 1)
            ptsEdge[k] = ptsEdge[k] + np.matlib.repmat(offset, ptsEdge[k].shape[0], 1)

        # Flatten 2-D indices to make 1-D in Column-order
        idxPts = []
        for k in np.arange(0, K, 1, dtype=int):
            ptsSub2Ind = np.ravel_multi_index([np.array(pts[k][0:,0]), np.array(pts[k][0:,1])], [bw.shape[0], bw.shape[1]])
            idxPts.append(ptsSub2Ind)

        # Check overlap of ellipse
        bGoodOverlap = GetOverlap(idxPts)
        if bGoodOverlap: # good overlap has pixel diff > 400pix
            bValidEllipse = True

    # Concatenate all the points of the ellipse(s) together
    ptsCombined = np.concatenate(pts[0:K])

    # Form binary image
    bw[np.array([ptsCombined[0:,0]]), np.array([ptsCombined[0:,1]])] = 1

    # Concatenate edges points of ellipse(s) together
    ptsEdgeCombined = np.concatenate(ptsEdge[0:K])

    # Identify crop boundaries
    ellipseBounds = np.where(bw == 1)
    minx = np.min(ellipseBounds[1]) - 10
    maxx = np.max(ellipseBounds[1]) + 10
    miny = np.min(ellipseBounds[0]) - 10
    maxy = np.max(ellipseBounds[0]) + 10

    # Crop boundaries on bw image
    bw = bw[miny:maxy, minx:maxx]
    bw = np.uint8(bw)

    # Normalize edge points according to new image boundaries
    ptsEdgeY = ptsEdgeCombined[0:, 0] - miny
    ptsEdgeX = ptsEdgeCombined[0:, 1] - minx
    ptsEdge = np.transpose(np.vstack((ptsEdgeY, ptsEdgeX)))
    return bw, ptsEdge # returns binary image (uint8, array) and true edge points(float, array)


# GenEllipse() accepts [a] mean values (array), [b] covariance matrix (array), [c] isocontour scalar(int),
# and [d] okay to 'fill ellipse' (bool). This function uses principle axes determined by the eigendecomposition
# of the convariance matrix to form one ellipse and generate its binary image. It returns the points and
def GenEllipse(mu, cv, c, bFill):
    import numpy as np
    import scipy.ndimage.morphology as bwmorph

    # Eigen-decomposition of covariance matrix
    [eigval, eigvec] = np.linalg.eig(cv)
    if len([eigval.shape]) == 1:
        eigval = np.sort(eigval)[::-1]
        order = np.argsort(eigval)[::-1]
    else:
        eigval = np.sort(np.diagonal(eigval))[::-1]
        order = np.argsort(np.diagonal(eigval))[::-1]
    eigvec = eigvec[0:, order]

    # Compute principle axes
    a1 = np.sqrt(eigval[0] * c * eigvec[0:, 0])
    a2 = np.sqrt(eigval[-1] * c * eigvec[0:, 1])

    # General ellipse equation (centered at mu) in polar coordinates
    theta = np.arange(0, 2 * np.pi, (2 * np.pi) / 1000, dtype=float) # range of theta
    x = np.array(mu[0] + a1[0] * np.sin(theta) + a2[0] * np.cos(theta))
    y = np.array(mu[1] + a1[1] * np.sin(theta) + a2[1] * np.cos(theta))

    # Clamp values to fall between [1, 1000]
    x = np.maximum(x, np.ones_like(x))
    y = np.maximum(y, np.ones_like(y))
    x = np.minimum(x, 1000 * np.ones_like(x))
    y = np.minimum(y, 1000 * np.ones_like(y))

    # Form binary image
    bw = np.zeros([1000, 1000])
    bw[np.array(np.round(x), dtype=int), np.array(np.round(y), dtype=int)] = 1

    # Fill image if the 'fill ellipse' bool is true
    if bFill:
        bw = bwmorph.binary_fill_holes(bw)

    # Collect points of generated ellipse
    idx = np.where(bw == 1)
    pts = np.transpose(np.vstack((idx[0], idx[1])))
    return pts, bw # returns points (float, array) and binary image of ellipse (int, array)


# GetRandomEllipsePts() accepts [a] mean and [b] size of the binary image. This generates one ellipse
# using random radii and random rotation. The ellipse is converted to a binary image.
def GetRandomEllipsePts(mu, IMSIZE):
    import numpy as np
    from numpy.matlib import repmat
    import scipy.ndimage.morphology as bwmorph

    # Initialize max radii parameters
    RADII_0 = 10
    RADII_1 = 20

    # Generate random radii
    r1 = RADII_0 + RADII_1 * np.random.rand()
    r2 = RADII_0 + RADII_1 * np.random.rand()

    # Populate covariance matrix [r1^2 0; 0 r2^2]
    cv = np.zeros([2, 2])
    cv[0,0] = r1**2
    cv[1,1] = r2**2

    # Generate ellipse points and edge points
    pts = GenEllipse(mu, cv, 1, 1)
    ptsEdge = GenEllipse(mu, cv, 1, 0)
    pts = pts[0]
    ptsEdge = ptsEdge[0]

    # Center points around origin (0, 0)
    pts = pts - np.matlib.repmat(mu, pts.shape[0], 1)
    ptsEdge = ptsEdge - np.matlib.repmat(mu, ptsEdge.shape[0], 1)

    # Rotate according to random theta by matrix multiplication of points with rotation matrix
    theta = 2 * np.pi * np.random.rand() # randomly generated theta
    rotate = np.zeros([2, 2])
    rotate[0, 0] = np.cos(theta)
    rotate[0, 1] = -np.sin(theta)
    rotate[1, 0] = np.sin(theta)
    rotate[1, 1] = np.cos(theta)
    pts = np.dot(pts, rotate)
    ptsEdge = np.dot(ptsEdge, rotate)

    # Re-center points around mu
    pts = pts + np.matlib.repmat(mu, pts.shape[0], 1)
    ptsEdge = ptsEdge + np.matlib.repmat(mu, ptsEdge.shape[0], 1)

    # Form binary image
    bw = np.zeros([IMSIZE, IMSIZE])
    roundPts = np.around(pts[0:,0:]) # must round points to index into binary image
    roundPts = roundPts.astype(int) # must be integer (not float)
    bw[roundPts[0:,0], roundPts[0:,1]] = 1
    bw = bwmorph.binary_fill_holes(bw) # fill holes generated from rounding

    # Identify points from newly formed binary image
    idx = np.where(bw==1)
    pts = np.transpose(np.vstack((idx[0], idx[1])))
    pts = np.array(pts)
    return pts, ptsEdge # returns ellipse points (int, array) and ellipse edge points (float, array)


# GetOverlap() accepts [a] linear index points (list) where each index of the list are the index points
# for each cluster. This using set difference theory to identify the non-intersecting points of each ellipse with
# the union of the remaining ellipse points. If the length of non-intersecting points is less than 400 pixels,
# the ellipses receive a bad overlap bool. Additionally, if the ellipses do not overlap, the ellipses receives
# a bad overlap bool.
def GetOverlap(idxPts):
    import numpy as np

    # Initialize good overlap boolean
    bGoodOverlap = True

    # Identify number of clusters
    K = len(idxPts)

    # Get intersection of each ellipse and remaining ellipse and evaluate overlap
    for k in np.arange(0, K, 1):
        kOther = np.setdiff1d(np.arange(0, K, 1), k)
        idxPts_other = []

        # Concatenate points of ellipses other than current k
        for i in kOther:
            idxPts_other = np.append(idxPts_other, idxPts[i])
        idxPts_test = idxPts[k]

        # Calculate intersection
        intersectPts = np.intersect1d(idxPts_other, idxPts_test)

        # Obtain non-intersecting points
        diffPts = idxPts_test.size - intersectPts.size

        # Evaluation of overlap
        if diffPts < 400:
            bGoodOverlap = False
            break
        if intersectPts.size == 0:
            bGoodOverlap = False
            break
    return bGoodOverlap # returns if ellipses have good/True or bad/False overlap (bool)

