% X = RejectionSim(n,pdfIm)
% 
% Simulate n points from the discrete empirical density pdfIm using the
% rejection method.

function X = RejectionSim(n,pdfIm)
    d = ndims(pdfIm);
    
    % Generate only near non-zero pixels
    bwIm = imdilate(pdfIm > 0, strel('square',3));
    validIdx = find(bwIm);
    
    coordCell = cell(1,d);
    [coordCell{:}] = ind2sub(size(pdfIm), validIdx);
    validCoords = [coordCell{:}];
    
    % Guarantee pdfIm is distribution and find probability of rejection c
    
    % g is the pdf of a uniform distribution over the valid pixels
    g = 1/length(validIdx);
    pdfIm = pdfIm / sum(pdfIm(:));
    
    c = max(pdfIm(:)) / g;
    
    lastIdx = 0;
    X = zeros(n,d);
    for i=1:ceil(2*c)
        U = rand(n,1);
        coordIdx = randi(length(validIdx),n,1);
        C = validCoords(coordIdx,:) + rand(n,d) - 0.5;
        
        cellCoord = num2cell(C,1);
        f = interpn(pdfIm, cellCoord{:}, 'linear');
        
        bKeep = (U <= f/(c*g));
        
        numValid = nnz(bKeep);
        writeNum = min(lastIdx+numValid,n)-lastIdx;
        
        validC = C(bKeep,:);
        X(lastIdx+(1:writeNum),:) = validC(1:writeNum,:);
        
        lastIdx = lastIdx + writeNum;
        if ( lastIdx >= n )
            break;
        end
    end
    
    if ( lastIdx < n )
        warning(['Only generated ' num2str(lastIdx) ' valid points!']);
    end
    
    swapCoord = [2 1 3:size(X,2)];
    X = X(:,swapCoord);
end
